%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Numerical integration of ODEs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all; clear; clc;
addpath(genpath('.'))
set( 0, 'defaultAxesTickLabelInterpreter', 'latex' );
set( 0, 'defaultLegendInterpreter',        'latex' );
set( 0, 'defaultTextInterpreter',          'latex' );
set( 0, 'defaultAxesFontSize', 18 );
%--------------------------------------------------------------------------

% Data of the problem;
f = @(t,y) (1-4/3*t)*y;   % RHS
y_exact_funhandle = @(t) exp( t - 2/3 * t.^2 );   % exact solution

T = 5;   % final time
y0 = 1;  % initial condition

% Numbers of grid points on the interval [0,T]:
N = logspace(2,4,3);
% N = logspace(2,6,5);

% Discretization parameters:
h = T./N;

% Define grey color for the reference lines:
grey = [ 0.6 0.6 0.6 ];

%--------------------------------------------------------------------------

% Number of refinements;
ref = length(N);

% Initializations of the arrays that will store the norms:
norEE_L2 = zeros(ref,1);
norHe_L2 = zeros(ref,1);
norRK_L2 = zeros(ref,1);
norEE = zeros(ref,1);
norHe = zeros(ref,1);
norRK = zeros(ref,1);

% Compute the errors of the discrete solutions:
for k = 1 : ref
    % Refine the mesh;
    t = linspace( 0, T, N(k)+1 )';

    % Compute the discrete solutions:
    y_exact  = y_exact_funhandle(t);
    [ y_EE ] = Explicit_Euler_NA( f, y0, t, h(k), N(k) );
    [ y_He ] = Heun_NA( f, y0, t, h(k), N(k) );
    [ y_RK ] = RK4_NA( f, y0, t, h(k), N(k) );
  
    % Compute the discrete L2 norm of the errors on [0,T]:
    norEE_L2(k) = sqrt(h(k)) * norm( (y_EE-y_exact), 2 );
    norHe_L2(k) = sqrt(h(k)) * norm( (y_He-y_exact), 2 );
    norRK_L2(k) = sqrt(h(k)) * norm( (y_RK-y_exact), 2 );
    
    % Compute the error at final time T:
    norEE(k) = abs( y_EE(end) - y_exact(end) )/abs(y_exact(end));
    norHe(k) = abs( y_He(end) - y_exact(end) )/abs(y_exact(end));
    norRK(k) = abs( y_RK(end) - y_exact(end) )/abs(y_exact(end));

end


%--------------------------------------------------------------------------
% Plots
%--------------------------------------------------------------------------
figure(1);

LineWidth = 2.5;

p1 = plot( t, y_EE, '-.g', 'LineWidth', LineWidth );
hold on;
p2 = plot( t, y_He, '--m', 'LineWidth', LineWidth );
p3 = plot( t, y_RK, '-b', 'LineWidth', LineWidth );
p4 = plot( t, y_exact, 'k', 'LineWidth', LineWidth );
axis( [ 0  T,   0  1.6 ])
grid on;
ylabel('$ y_{h} $');
xlabel('$ t $');
hL = legend( [ p1, p2, p3, p4 ], ...
    { 'Explicit Euler', 'Heun', 'RK', 'exact solution' }, ...
    'Location', 'NorthEast' );
set( hL, 'Orientation', 'vertical' );

%--------------------------------------------------------------------------
% Plot the error of the discrete solutions
figure(2);

LineWidthReferenceLines = 1.25;

p1 = loglog( h, norEE_L2, 'o-.g', 'LineWidth', LineWidth );
hold on;
p2 = loglog( h, norHe_L2, 'o--m', 'LineWidth', LineWidth );
p3 = loglog( h, norRK_L2, 'o-b', 'LineWidth', LineWidth );
pO1 = loglog( h, 2.5e-1*h, '-.', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
pO2 = loglog( h, 1e-1*h.^2, '--', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
pO4 = loglog( h, 1e-2*h.^4, '-', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
grid on;
xlabel('$h$');
ylabel('$ \| y - y_{h} \|_{2} $');
hL = legend( [ p1, p2, p3, pO1, pO2, pO4 ], ...
    { 'Explicit Euler', 'Heun', 'RK', ...
    '$ \mathcal{O}(h) $', '$ \mathcal{O}(h^2) $', '$ \mathcal{O}(h^4) $' }, ...
    'Location','SouthEast');
set( hL, 'Orientation', 'vertical' );

%--------------------------------------------------------------------------
figure(3);
p1 = loglog( h, norEE, 'o-.g', 'LineWidth', LineWidth );
hold on;
p2 = loglog( h, norHe, 'o--m', 'LineWidth', LineWidth );
p3 = loglog( h, norRK, 'o-b', 'LineWidth', LineWidth );
pO1 = loglog( h, 1e1*h, '-.', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
pO2 = loglog( h, 5e0*h.^2, '--', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
pO4 = loglog( h, 1e0*h.^4, '-', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
grid on;
xlabel('$h$');
ylabel('$ | y(T) - y_{h}(T) |/| y(T) | $');
hL = legend( [ p1, p2, p3, pO1, pO2, pO4 ], ...
    { 'Explicit Euler', 'Heun', 'RK', ...
    '$ \mathcal{O}(h) $', '$ \mathcal{O}(h^2) $', '$ \mathcal{O}(h^4) $' }, ...
    'Location', 'SouthEast' );
set( hL, 'Orientation', 'vertical' );
