%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Numerical integration of ODEs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all; clear; clc;   % this is how I usually start a MATLAB script

% Some options for plotting purposes:
set( 0, 'defaultAxesTickLabelInterpreter', 'latex' );
set( 0, 'defaultLegendInterpreter',        'latex' );
set( 0, 'defaultTextInterpreter',          'latex' );
set( 0, 'defaultAxesFontSize', 18 );

LineWidth = 2.5;   % line width for the solution and error plots
LineWidthReferenceLines = 1.25;   % line width for the reference lines in the error plot

% Define grey color for the reference lines:
grey = [ 0.6 0.6 0.6 ];

%--------------------------------------------------------------------------

% Data of the problem;
f = @(t,y) (1-4/3*t)*y;   % RHS
y_exact_funhandle = @(t) exp( t - 2/3 * t.^2 );   % exact solution

T = 5;   % final time
y0 = 1;  % initial condition

% Number of refinements;
ref = 5;

% Numbers of grid points on the interval [0,T]:
N = logspace(2,ref+1,ref);   % creates a vector with 5 powers of 10, from 10^2 to 10^(ref+1)

% Vector of discretization parameters:
h = T./N;

%--------------------------------------------------------------------------

% Initializations of the arrays that will store the norms:
norEE_L2 = zeros(ref,1);
norEE = zeros(ref,1);

% Compute the errors of the discrete solutions:
for k = 1:ref
    % Define the mesh;
    t = linspace( 0, T, N(k)+1 )';

    % Compute the discrete solutions:
    y_exact  = y_exact_funhandle(t);
    [ y_EE ] = Explicit_Euler_NA( f, y0, t, h(k), N(k) );

    % Compute the discrete L2 norm of the errors on [0,T]:
    norEE_L2(k) = sqrt(h(k)) * norm( (y_EE-y_exact), 2 );

    % Compute the error at final time T:
    norEE(k) = abs( y_EE(end) - y_exact(end) )/abs(y_exact(end));

end


%--------------------------------------------------------------------------
% Plots
%--------------------------------------------------------------------------
% Plot the numerical solution and the exact solution
figure(1);
plot( t, y_EE, '-.r', 'LineWidth', LineWidth );
hold on;
plot( t, y_exact, 'k', 'LineWidth', LineWidth );
grid on;
ylabel('$ y_{h} $, $ y_{\mathrm{exact}} $');
xlabel('$ t $');
legend( 'Explicit Euler', 'exact solution', 'Location', 'NorthEast' );
title( 'Numerical and exact solution' )

%--------------------------------------------------------------------------
% Plot the error of the discrete solutions, calculated with the discrete
% L2-norm
figure(2);
loglog( h, norEE_L2, 'o-.r', 'LineWidth', LineWidth );
hold on;
loglog( h, h, '-.', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
grid on;
xlabel('$h$');
ylabel('$ \| y_{\mathrm{exact}} - y_{h} \|_{L^{2}([0,T])} $');
legend( 'Explicit Euler', '$ \mathcal{O}(h) $', 'Location', 'SouthEast');
title( 'Error in the discrete $L^{2}$-norm' )

%--------------------------------------------------------------------------
% Plot the (relative) error of the discrete solutions at final time T
figure(3);
loglog( h, norEE, 'o-.r', 'LineWidth', LineWidth );
hold on;
loglog( h, h, '-.', 'Color', grey, 'LineWidth', LineWidthReferenceLines );
grid on;
xlabel('$h$');
ylabel('$ | y_{\mathrm{exact}}(T) - y_{h}(T) |/| y_{\mathrm{exact}}(T) | $');
legend( 'Explicit Euler', '$ \mathcal{O}(h) $', 'Location', 'SouthEast' );
title( 'Relative error in absolute value at the final time $T$' )