%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Stability Regions of RK4 and Dormand-Prince (RK45) via the "ODE method".

% Created:     2025.12.04
% Last change: 2025.12.11
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all; clear; clc;
addpath(genpath('.'))
% Set latex as the intepreter for the text in the plot and default axes
% fontsize
set( 0, 'defaultAxesTickLabelInterpreter', 'latex' );
set( 0, 'defaultLegendInterpreter',        'latex' );
set( 0, 'defaultTextInterpreter',          'latex' );
set( 0, 'defaultAxesFontSize', 16 );   % or 14

%--------------------------------------------------------------------------
% 1) RK4
R4  = @(z) 1 + z + z^2/2 + z^3/6 + z^4/24;   % stability function for RK4
Rp4 = @(z) 1 + z + z^2/2 + z^3/6;            % its derivative

% Integration parameters
tmax = 8*pi;

% Newton initial guess (R(z)=1)
z = 0;
for k = 1:10
    z = z - (R4(z) - 1) / Rp4(z);
end
z0_4 = z;

% ODE for the RK4 stability region
odefun4 = @(t,z) 1i*exp(1i*t) / Rp4(z);
opts = odeset( 'RelTol', 1e-10, 'AbsTol', 1e-12 );   % options for the ode45 solver

[ t4, Z4 ] = ode45( odefun4, [0 tmax], z0_4, opts );


%--------------------------------------------------------------------------
% 2) Dormand-Prince (RK45)
% Butcher tableau of Dormand-Prince 5(4)
A = [...
    0           0           0           0           0           0           0;
    1/5         0           0           0           0           0           0;
    3/40        9/40        0           0           0           0           0;
    44/45      -56/15       32/9        0           0           0           0;
    19372/6561 -25360/2187  64448/6561 -212/729     0           0           0;
    9017/3168  -355/33      46732/5247  49/176     -5103/18656  0           0;
    35/384      0           500/1113    125/192    -2187/6784   11/84       0 ];

% b vector (the 5th-order solution)
b = [ 35/384; 0; 500/1113; 125/192; -2187/6784; 11/84; 0 ];

e = ones(7,1);
I7 = eye(7);

% Stability function R(z) = 1 + z b^T (I - z A)^{-1} e
Rdp = @(z) arrayfun(@(zz) 1 + zz*b'*( (I7 - zz*A) \ e ), z);

% Derivative of R(z): numeric differentiation (sufficiently accurate)
hz = 1e-8;   % discretization parameter for finite difference approximation
Rpdp = @(z) (Rdp(z+hz) - Rdp(z-hz)) / (2*hz);

% Initial point R(z)=1
z = 0;
for k = 1:10
    z = z - (Rdp(z) - 1) / Rpdp(z);
end
z0_dp = z;

% ODE for the Dormand-Prince stability region
odefundp = @(t,z) 1i*exp(1i*t) / Rpdp(z);

[ tDP, ZDP ] = ode45( odefundp, [0 tmax], z0_dp, opts );


%--------------------------------------------------------------------------
% 3) Plot results
figure;
hold on;
grid on;

plot([-6.5, 2], [0, 0], '--', 'Color', 'k', 'LineWidth', 1.5);  % x axis
plot([0, 0], [-3.5, 3.5], '--', 'Color', 'k', 'LineWidth', 1.5);  % y axis

p1 = plot(real(Z4),  imag(Z4),  'b-', 'LineWidth', 2);
p2 = plot(real(ZDP), imag(ZDP), 'r-', 'LineWidth', 2);
axis equal;
xlabel('Re(z)');
ylabel('Im(z)');
title('Stability regions');

xlim( [ -6.5, 2 ] );
ylim( [ -3.5, 3.5 ] );

legend( [ p1, p2 ], 'RK4', 'Dormand-Prince 5(4)', 'Location', 'NW' );
