function [ gridpoints, sol ] = RK45( fun, t0, tf, y0, h0, Tol )
% function [ gridpoints, sol ] = RK45( fun, t0, tf, y0, h0, Tol )
% Code originally written by Nicola Guglielmi.
% Last change: 2025.12.04 by Marco Sutti.

% Coefficients of the Dormand-Prince method:
A = [      0,             0,            0,          0,             0,            0;
         1/5,             0,            0,          0,             0,            0;
        3/40,          9/40,            0,          0,             0,            0;
       44/45,        -56/15,         32/9,          0,             0,            0;
  19372/6561,   -25360/2187,   64448/6561,   -212/729,             0,            0;
   9017/3168,       -355/33,   46732/5247,     49/176,   -5103/18656,            0;
      35/384,             0,     500/1113,    125/192,    -2187/6784,        11/84 ];

c = [ 1/5;   3/10;   4/5;   8/9;   1;   1 ];
 
w1 = [     35/384,   0,     500/1113,   125/192,      -2187/6784,      11/84 ];
w2 = [ 5179/57600,   0,   7571/16695,   393/640,   -92097/339200,   187/2100,   1/40 ];


% Initializations
y_hat0 = y0;   % the 4th-order embedded method is initialized like the 5th-order method
h = h0;
d = length(y0);   % 2 for the Brusselator problem
sol = zeros(d,1);
sol(:,1) = y0;
gridpoints = zeros(1,1);
gridpoints(1) = t0;
k_rej = 0;   % counter for the number of rejected steps
kappas = zeros(2,7);   % 2D array that will store the kappa coefficients

while t0 < tf

    kappas(:,1) = feval(fun, t0, y0);
    for i=2:7
        kappas(:,i) = feval( fun, t0 + c(i-1)*h, y0 + h*(sum( A(i,1:i-1).*kappas(:,1:i-1), 2) ) );
    end

    y1 = y0 + h * sum( w1.*kappas(:,1:6), 2 );    % numerical solution with the 5th-order method 
    y_hat1 = y_hat0 + h * sum( w2.*kappas, 2 );   % numerical solution with the 4th-order embedded method
    
    sc = 1 + max(abs(y0), abs(y1)); 
    err = (y1 - y_hat1)./sc;
    err = sqrt((1/d)*norm(err)^2);

    if err <= Tol   % the step size h is accepted
        ns = size(sol,2);
        sol(:, ns+1) = y1;   % store the solution at the current time step
        gridpoints(ns+1) = t0 + h;
        y0 = y1;
        y_hat0 = y1;
        t0 = t0 + h;
        h1 = 0.9*h*(Tol/err)^(1/5);
        % hopt = min( 5*h, h1 );   % avoid too big h
        h = min( h1, (tf - t0) );
    else   % the step size h is rejected
        h = 0.9*h*(Tol/err)^(1/5);
        % h = max( 0.2*h, h1 );    % avoid too small h
        k_rej = k_rej + 1;   % increase counter for the number of rejected steps
    end
end

fprintf('Number of rejected steps = %d.\n', k_rej )

end